<?php

require_once 'config.php';

/**
 * Initialize the database and create tables
 */
function init_db(): void {
    $db = new PDO('sqlite:' . DATABASE);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Create files table
    $db->exec("
        CREATE TABLE IF NOT EXISTS files (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            link_code TEXT UNIQUE,
            file_data TEXT,
            view_count INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    // Create users table
    $db->exec("
        CREATE TABLE IF NOT EXISTS users (
            user_id INTEGER PRIMARY KEY,
            join_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    // Create channels table
    $db->exec("
        CREATE TABLE IF NOT EXISTS channels (
            chat_id INTEGER PRIMARY KEY,
            channel_link TEXT
        )
    ");
    
    // Create likes table
    $db->exec("
        CREATE TABLE IF NOT EXISTS likes (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            link_code TEXT,
            user_id INTEGER,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(link_code, user_id)
        )
    ");
}

/**
 * Get database connection
 */
function get_db(): PDO {
    $db = new PDO('sqlite:' . DATABASE);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    return $db;
}

/**
 * Add a file record
 */
function add_file_record(string $link_code, array $file_data): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT INTO files (link_code, file_data) VALUES (?, ?)");
    $stmt->execute([$link_code, json_encode($file_data)]);
}

/**
 * Get file record by link code
 */
function get_file_record(string $link_code): ?array {
    $db = get_db();
    $stmt = $db->prepare("SELECT id, file_data, view_count FROM files WHERE link_code = ?");
    $stmt->execute([$link_code]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($row) {
        return [
            'id' => $row['id'],
            'file_data' => json_decode($row['file_data'], true),
            'view_count' => $row['view_count']
        ];
    }
    return null;
}

/**
 * Increment view count for a file
 */
function increment_view_count(string $link_code): void {
    $db = get_db();
    $stmt = $db->prepare("UPDATE files SET view_count = view_count + 1 WHERE link_code = ?");
    $stmt->execute([$link_code]);
}

/**
 * Add a user to the database
 */
function add_user(int $user_id): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT OR IGNORE INTO users (user_id) VALUES (?)");
    $stmt->execute([$user_id]);
}

/**
 * Get all users
 */
function get_all_users(): array {
    $db = get_db();
    $stmt = $db->query("SELECT user_id FROM users");
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Get user count
 */
function get_user_count(): int {
    $db = get_db();
    $stmt = $db->query("SELECT COUNT(*) FROM users");
    return (int)$stmt->fetchColumn();
}

/**
 * Add a channel
 */
function add_channel(int $chat_id, string $channel_link): void {
    $db = get_db();
    $stmt = $db->prepare("INSERT OR REPLACE INTO channels (chat_id, channel_link) VALUES (?, ?)");
    $stmt->execute([$chat_id, $channel_link]);
}

/**
 * Remove a channel
 */
function remove_channel(int $chat_id): void {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM channels WHERE chat_id = ?");
    $stmt->execute([$chat_id]);
}

/**
 * Get all channels
 */
function get_channels(): array {
    $db = get_db();
    $stmt = $db->query("SELECT chat_id, channel_link FROM channels");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $channels = [];
    foreach ($rows as $row) {
        $channels[] = [
            'chat_id' => $row['chat_id'],
            'channel_link' => $row['channel_link']
        ];
    }
    return $channels;
}

/**
 * Add a like
 */
function add_like(string $link_code, int $user_id): bool {
    $db = get_db();
    try {
        $stmt = $db->prepare("INSERT INTO likes (link_code, user_id) VALUES (?, ?)");
        $stmt->execute([$link_code, $user_id]);
        return true;
    } catch (PDOException $e) {
        // User already liked this link (unique constraint violation)
        return false;
    }
}

/**
 * Remove a like
 */
function remove_like(string $link_code, int $user_id): bool {
    $db = get_db();
    $stmt = $db->prepare("DELETE FROM likes WHERE link_code = ? AND user_id = ?");
    $stmt->execute([$link_code, $user_id]);
    return $stmt->rowCount() > 0;
}

/**
 * Get like count for a link
 */
function get_like_count(string $link_code): int {
    $db = get_db();
    $stmt = $db->prepare("SELECT COUNT(*) FROM likes WHERE link_code = ?");
    $stmt->execute([$link_code]);
    return (int)$stmt->fetchColumn();
}

/**
 * Check if user has liked a link
 */
function has_user_liked(string $link_code, int $user_id): bool {
    $db = get_db();
    $stmt = $db->prepare("SELECT 1 FROM likes WHERE link_code = ? AND user_id = ?");
    $stmt->execute([$link_code, $user_id]);
    return $stmt->fetch() !== false;
}

?>

