<?php

require_once 'config.php';
require_once 'database.php';
require_once 'functions.php';

// Get incoming update from Telegram
$content = file_get_contents('php://input');
$update = json_decode($content, true);

// Log the update for debugging
if (DEBUG_MODE) {
    file_put_contents(__DIR__ . '/updates.log', date('Y-m-d H:i:s') . " - " . $content . "\n", FILE_APPEND);
}

// Exit if no update received
if (!$update) {
    http_response_code(200);
    exit;
}

// Handle different update types
if (isset($update['message'])) {
    handle_message($update['message']);
} elseif (isset($update['callback_query'])) {
    handle_callback_query($update['callback_query']);
}

http_response_code(200);

/**
 * Handle incoming messages
 */
function handle_message(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'] ?? '';
    
    // Handle /start command
    if (isset($message['entities']) && $message['entities'][0]['type'] === 'bot_command' && strpos($text, '/start') === 0) {
        handle_start_command($message);
        return;
    }
    
    // Handle /cancel command for admins
    if ($text === '/cancel' && is_admin($user_id)) {
        $state = get_admin_state($user_id);
        if ($state) {
            $mode = $state['mode'] ?? '';
            clear_admin_state($user_id);
            
            if ($mode === 'add_channel') {
                send_message($chat_id, 'عملیات لغو شد.', channel_lock_keyboard());
            } else {
                send_message($chat_id, 'عملیات لغو شد.', admin_panel_keyboard());
            }
        }
        return;
    }
    
    // Handle /done command for admins
    if ($text === '/done' && is_admin($user_id)) {
        $state = get_admin_state($user_id);
        if ($state && $state['mode'] === 'upload') {
            if (empty($state['files'])) {
                send_message($chat_id, 'هیچ فایلی دریافت نشده است.');
                return;
            }
            
            $link_code = generate_link_code();
            add_file_record($link_code, $state['files']);
            
            $bot_username = get_bot_username();
            $link = "https://t.me/{$bot_username}?start={$link_code}";
            
            send_message($chat_id, "فایل(ها) ذخیره شدند.\nلینک دریافت فایل:\n{$link}");
            clear_admin_state($user_id);
            send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
        }
        return;
    }
    
    // Handle admin state-based messages
    if (is_admin($user_id)) {
        handle_admin_state_message($message);
    }
}

/**
 * Handle /start command
 */
function handle_start_command(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'];
    
    // Add user to database if not admin
    if (!is_admin($user_id)) {
        add_user($user_id);
    }
    
    // Extract payload (link code) from /start command
    $parts = explode(' ', $text, 2);
    
    if (count($parts) > 1) {
        $payload = $parts[1];
        $file_record = get_file_record($payload);
        
        if (!$file_record) {
            send_message($chat_id, 'لینک نامعتبر است!');
            return;
        }
        
        // Check if user is member of all required channels
        $not_member = check_user_membership($user_id);
        
        if (!empty($not_member)) {
            send_message(
                $chat_id,
                'برای دریافت فایل باید در کانال‌های زیر عضو شوید:',
                user_join_keyboard($payload)
            );
            return;
        }
        
        // Send files to user
        send_files_to_user($chat_id, $user_id, $payload, $file_record);
        
    } else {
        // No payload - show admin panel or welcome message
        if (is_admin($user_id)) {
            send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
        } else {
            send_message($chat_id, 'سلام! برای دریافت فایل، لطفاً از لینک معتبر استفاده کنید.');
        }
    }
}

/**
 * Send files to user
 */
function send_files_to_user(int $chat_id, int $user_id, string $link_code, array $file_record): void {
    $file_data = $file_record['file_data'];
    
    foreach ($file_data as $item) {
        copy_message($chat_id, $item['chat_id'], $item['message_id']);
    }
    
    increment_view_count($link_code);
    
    send_message(
        $chat_id,
        "فایل ارسال شد. تعداد مشاهده: " . ($file_record['view_count'] + 1),
        file_sent_keyboard($link_code, $user_id)
    );
}

/**
 * Handle admin state-based messages
 */
function handle_admin_state_message(array $message): void {
    $user_id = $message['from']['id'];
    $chat_id = $message['chat']['id'];
    $message_id = $message['message_id'];
    $text = $message['text'] ?? '';
    
    $state = get_admin_state($user_id);
    if (!$state) {
        return;
    }
    
    $mode = $state['mode'];
    
    // Handle upload mode
    if ($mode === 'upload') {
        $state['files'][] = [
            'chat_id' => $chat_id,
            'message_id' => $message_id
        ];
        set_admin_state($user_id, $state);
        send_message($chat_id, 'فایل ذخیره شد.');
    }
    
    // Handle broadcast modes
    elseif ($mode === 'broadcast_copy' || $mode === 'broadcast_forward') {
        $users = get_all_users();
        $success = 0;
        $failure = 0;
        
        foreach ($users as $uid) {
            try {
                if ($mode === 'broadcast_copy') {
                    copy_message($uid, $chat_id, $message_id);
                } else {
                    forward_message($uid, $chat_id, $message_id);
                }
                $success++;
            } catch (Exception $e) {
                $failure++;
            }
        }
        
        send_message($chat_id, "ارسال همگانی انجام شد.\nموفق: {$success}\nناموفق: {$failure}");
        clear_admin_state($user_id);
        send_message($chat_id, 'پنل ادمین:', admin_panel_keyboard());
    }
    
    // Handle add channel mode
    elseif ($mode === 'add_channel') {
        $step = $state['step'];
        
        if ($step === 'chat_id') {
            if (is_numeric($text)) {
                $channel_chat_id = (int)$text;
                
                // Verify bot has access to the channel
                $chat = get_chat($channel_chat_id);
                
                if ($chat) {
                    $state['chat_id'] = $channel_chat_id;
                    $state['step'] = 'channel_link';
                    set_admin_state($user_id, $state);
                    
                    send_message(
                        $chat_id,
                        "Chat ID تأیید شد. حالا لطفاً لینک کانال را ارسال کنید.\nمثال: https://t.me/yourchannel یا @yourchannel"
                    );
                } else {
                    send_message(
                        $chat_id,
                        "خطا: ربات به این کانال دسترسی ندارد.\n" .
                        "لطفاً مطمئن شوید که:\n" .
                        "1. Chat ID صحیح است\n" .
                        "2. ربات ادمین کانال است\n" .
                        "3. ربات به کانال دسترسی دارد\n\n" .
                        "لطفاً دوباره Chat ID را ارسال کنید یا /cancel را بزنید."
                    );
                }
            } else {
                send_message($chat_id, "لطفاً یک Chat ID معتبر ارسال کنید.\nمثال: -1001234567890");
            }
        }
        
        elseif ($step === 'channel_link') {
            if (preg_match('/^(https?:\/\/t\.me\/|@)/', $text)) {
                $channel_link = $text;
                
                if (strpos($channel_link, '@') === 0) {
                    $channel_link = 'https://t.me/' . substr($channel_link, 1);
                }
                
                add_channel($state['chat_id'], $channel_link);
                send_message($chat_id, '✅ کانال با موفقیت اضافه شد!');
                clear_admin_state($user_id);
                send_message($chat_id, 'مدیریت قفل کانال:', channel_lock_keyboard());
            } else {
                send_message(
                    $chat_id,
                    "لطفاً یک لینک معتبر کانال ارسال کنید.\nمثال: https://t.me/yourchannel یا @yourchannel"
                );
            }
        }
    }
}

/**
 * Handle callback queries (inline button clicks)
 */
function handle_callback_query(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];
    
    // Handle like button
    if (strpos($data, 'like:') === 0) {
        $link_code = substr($data, 5);
        $like_status = has_user_liked($link_code, $user_id);
        
        if ($like_status) {
            if (remove_like($link_code, $user_id)) {
                answer_callback_query($callback_id, 'لایک شما حذف شد');
            }
        } else {
            if (add_like($link_code, $user_id)) {
                answer_callback_query($callback_id, 'با تشکر از لایک شما!');
            } else {
                answer_callback_query($callback_id, 'قبلا لایک کرده‌اید!');
            }
        }
        
        edit_message_reply_markup($chat_id, $message_id, file_sent_keyboard($link_code, $user_id));
        return;
    }
    
    // Handle admin panel callbacks
    if (strpos($data, 'admin_') === 0) {
        if (!is_admin($user_id)) {
            answer_callback_query($callback_id, 'شما اجازه دسترسی به این بخش را ندارید.', true);
            return;
        }
        
        handle_admin_callback($callback_query);
        return;
    }
    
    // Handle join check callback
    if (strpos($data, 'join_check:') === 0) {
        $link_code = substr($data, 11);
        
        $not_member = check_user_membership($user_id);
        
        if (!empty($not_member)) {
            answer_callback_query($callback_id, 'هنوز عضو همه کانال‌ها نشده‌اید.', true);
        } else {
            $file_record = get_file_record($link_code);
            
            if (!$file_record) {
                answer_callback_query($callback_id, 'لینک نامعتبر است.', true);
                return;
            }
            
            send_files_to_user($chat_id, $user_id, $link_code, $file_record);
            answer_callback_query($callback_id, 'فایل ارسال شد.');
        }
        return;
    }
    
    // Handle channel management callbacks
    if (strpos($data, 'channel_') === 0) {
        handle_channel_callback($callback_query);
        return;
    }
}

/**
 * Handle admin panel callbacks
 */
function handle_admin_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];
    
    if ($data === 'admin_upload') {
        set_admin_state($user_id, ['mode' => 'upload', 'files' => []]);
        edit_message_text(
            $chat_id,
            $message_id,
            'لطفا فایل (یا پیام) مورد نظر برای آپلود را ارسال کنید. پس از اتمام، دستور /done را بفرستید.'
        );
    }
    
    elseif ($data === 'admin_stats') {
        $count = get_user_count();
        answer_callback_query($callback_id, "تعداد کاربران: {$count}", true);
    }
    
    elseif ($data === 'admin_broadcast_copy') {
        set_admin_state($user_id, ['mode' => 'broadcast_copy']);
        edit_message_text($chat_id, $message_id, 'پیام مورد نظر را برای ارسال همگانی (copy) ارسال کنید.');
    }
    
    elseif ($data === 'admin_broadcast_forward') {
        set_admin_state($user_id, ['mode' => 'broadcast_forward']);
        edit_message_text($chat_id, $message_id, 'پیام مورد نظر را برای ارسال همگانی (forward) ارسال کنید.');
    }
    
    elseif ($data === 'admin_lockchannel') {
        edit_message_text($chat_id, $message_id, 'مدیریت قفل کانال:', channel_lock_keyboard());
    }
    
    answer_callback_query($callback_id);
}

/**
 * Handle channel management callbacks
 */
function handle_channel_callback(array $callback_query): void {
    $callback_id = $callback_query['id'];
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $data = $callback_query['data'];
    
    if (strpos($data, 'channel_info_') === 0) {
        $channel_chat_id = (int)substr($data, 13);
        $channels = get_channels();
        
        $channel = null;
        foreach ($channels as $ch) {
            if ($ch['chat_id'] == $channel_chat_id) {
                $channel = $ch;
                break;
            }
        }
        
        if ($channel) {
            $text = "کانال:\nChat ID: {$channel['chat_id']}\nLink: {$channel['channel_link']}";
            $keyboard = [
                'inline_keyboard' => [
                    [['text' => 'حذف کانال', 'callback_data' => "channel_remove_{$channel_chat_id}"]]
                ]
            ];
            edit_message_text($chat_id, $message_id, $text, $keyboard);
        } else {
            answer_callback_query($callback_id, 'کانال یافت نشد.', true);
        }
    }
    
    elseif (strpos($data, 'channel_remove_') === 0) {
        $channel_chat_id = (int)substr($data, 15);
        remove_channel($channel_chat_id);
        answer_callback_query($callback_id, 'کانال حذف شد.', true);
        edit_message_text($chat_id, $message_id, 'کانال حذف شد.', channel_lock_keyboard());
    }
    
    elseif ($data === 'channel_add') {
        set_admin_state($user_id, ['mode' => 'add_channel', 'step' => 'chat_id']);
        edit_message_text(
            $chat_id,
            $message_id,
            "لطفاً Chat ID کانال را ارسال کنید.\nمثال: -1001234567890"
        );
    }
}

?>

