<?php

require_once 'config.php';
require_once 'database.php';

/**
 * Generate a random link code
 */
function generate_link_code(int $length = 8): string {
    $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $code = '';
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[random_int(0, strlen($characters) - 1)];
    }
    return $code;
}

/**
 * Make Telegram API request
 */
function telegram_request(string $method, array $data = []): array {
    $url = TELEGRAM_API . $method;
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code !== 200) {
        error_log("Telegram API error: HTTP $http_code - $response");
    }
    
    return json_decode($response, true) ?? [];
}

/**
 * Send message to Telegram
 */
function send_message(int $chat_id, string $text, ?array $reply_markup = null): void {
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup !== null) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    
    telegram_request('sendMessage', $data);
}

/**
 * Edit message text
 */
function edit_message_text(int $chat_id, int $message_id, string $text, ?array $reply_markup = null): void {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($reply_markup !== null) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    
    telegram_request('editMessageText', $data);
}

/**
 * Edit message reply markup only
 */
function edit_message_reply_markup(int $chat_id, int $message_id, array $reply_markup): void {
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'reply_markup' => json_encode($reply_markup)
    ];
    
    telegram_request('editMessageReplyMarkup', $data);
}

/**
 * Answer callback query
 */
function answer_callback_query(string $callback_query_id, string $text = '', bool $show_alert = false): void {
    $data = [
        'callback_query_id' => $callback_query_id,
        'text' => $text,
        'show_alert' => $show_alert
    ];
    
    telegram_request('answerCallbackQuery', $data);
}

/**
 * Copy message
 */
function copy_message(int $chat_id, int $from_chat_id, int $message_id): ?int {
    $data = [
        'chat_id' => $chat_id,
        'from_chat_id' => $from_chat_id,
        'message_id' => $message_id
    ];
    
    $response = telegram_request('copyMessage', $data);
    return $response['result']['message_id'] ?? null;
}

/**
 * Forward message
 */
function forward_message(int $chat_id, int $from_chat_id, int $message_id): void {
    $data = [
        'chat_id' => $chat_id,
        'from_chat_id' => $from_chat_id,
        'message_id' => $message_id
    ];
    
    telegram_request('forwardMessage', $data);
}

/**
 * Get chat member
 */
function get_chat_member(int $chat_id, int $user_id): ?array {
    $data = [
        'chat_id' => $chat_id,
        'user_id' => $user_id
    ];
    
    $response = telegram_request('getChatMember', $data);
    return $response['result'] ?? null;
}

/**
 * Get chat
 */
function get_chat(int $chat_id): ?array {
    $data = ['chat_id' => $chat_id];
    $response = telegram_request('getChat', $data);
    return $response['result'] ?? null;
}

/**
 * Get bot username
 */
function get_bot_username(): string {
    $response = telegram_request('getMe');
    return $response['result']['username'] ?? 'bot';
}

/**
 * Check if user is member of all channels
 */
function check_user_membership(int $user_id): array {
    $channels = get_channels();
    $not_member = [];
    
    foreach ($channels as $channel) {
        $member = get_chat_member($channel['chat_id'], $user_id);
        if ($member === null || in_array($member['status'], ['left', 'kicked'])) {
            $not_member[] = $channel;
        }
    }
    
    return $not_member;
}

/**
 * Create admin panel keyboard
 */
function admin_panel_keyboard(): array {
    return [
        'inline_keyboard' => [
            [['text' => '📦 اپلود فایل', 'callback_data' => 'admin_upload']],
            [['text' => '📊 آمار ربات', 'callback_data' => 'admin_stats']],
            [['text' => '💌 ارسال همگانی', 'callback_data' => 'admin_broadcast_copy']],
            [['text' => '📬فوروارد همگانی', 'callback_data' => 'admin_broadcast_forward']],
            [['text' => '🔮قفل کانال', 'callback_data' => 'admin_lockchannel']]
        ]
    ];
}

/**
 * Create channel lock keyboard
 */
function channel_lock_keyboard(): array {
    $channels = get_channels();
    $buttons = [];
    
    foreach ($channels as $channel) {
        $buttons[] = [['text' => "کانال: {$channel['chat_id']}", 'callback_data' => "channel_info_{$channel['chat_id']}"]];
    }
    
    $buttons[] = [['text' => '➕ افزودن کانال', 'callback_data' => 'channel_add']];
    
    return ['inline_keyboard' => $buttons];
}

/**
 * Create user join keyboard
 */
function user_join_keyboard(string $link_code): array {
    $channels = get_channels();
    $buttons = [];
    
    foreach ($channels as $channel) {
        $chat = get_chat($channel['chat_id']);
        $channel_name = $chat['title'] ?? "کانال";
        $buttons[] = [['text' => $channel_name, 'url' => $channel['channel_link']]];
    }
    
    $buttons[] = [['text' => 'عضو شدم', 'callback_data' => "join_check:$link_code"]];
    
    return ['inline_keyboard' => $buttons];
}

/**
 * Create file sent keyboard with like button
 */
function file_sent_keyboard(string $link_code, int $user_id): array {
    $like_status = has_user_liked($link_code, $user_id);
    $like_count = get_like_count($link_code);
    
    $like_text = $like_status ? "❤️" : "🤍";
    $like_text .= " ($like_count)";
    
    return [
        'inline_keyboard' => [
            [['text' => $like_text, 'callback_data' => "like:$link_code"]]
        ]
    ];
}

/**
 * Load admin states from session file
 */
function load_admin_states(): array {
    $file = __DIR__ . '/admin_states.json';
    if (file_exists($file)) {
        $content = file_get_contents($file);
        return json_decode($content, true) ?? [];
    }
    return [];
}

/**
 * Save admin states to session file
 */
function save_admin_states(array $states): void {
    $file = __DIR__ . '/admin_states.json';
    file_put_contents($file, json_encode($states, JSON_PRETTY_PRINT));
}

/**
 * Get admin state for a user
 */
function get_admin_state(int $user_id): ?array {
    $states = load_admin_states();
    return $states[$user_id] ?? null;
}

/**
 * Set admin state for a user
 */
function set_admin_state(int $user_id, array $state): void {
    $states = load_admin_states();
    $states[$user_id] = $state;
    save_admin_states($states);
}

/**
 * Clear admin state for a user
 */
function clear_admin_state(int $user_id): void {
    $states = load_admin_states();
    unset($states[$user_id]);
    save_admin_states($states);
}

/**
 * Check if user is admin
 */
function is_admin(int $user_id): bool {
    return in_array($user_id, ADMIN_IDS);
}

?>

